// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2001
//
// Routines for manipulating a Sol virtual tape.

/*
PROBLEM:

We want to have a set of routines for accessing and manipulating a
virtual cassette tape that can represent the Sol tape behavior as
much as possible.  These routines should be as independent of the
emulator as possible, as it may be desirable to use these functions
for other utility applications, such as cataloging; adding and
removing files; saving to real tape; capturing from real tape.

Another implication of this is that the tape has no sense of real
time.  It is up to the caller to read or write bytes as needed.

The external tape representation is a (mostly) human readable file
format, stored as 7b ASCII.  The details of this format can be
read in the docs/tape.txt file.

The internal tape representation for the tape is somewhat complicated
by the fact that there are two speeds of recording.

Rather than having different segments for different speeds (which
is OK until you start thinking about what has to happen to these
segments when recording one speed bleeds into or is in the middle
of the other speed -- there would be a lot of boundary cases where
segments get split or joined or deleted), the thought is to
have a slightly wasteful but more uniform representation.

ATOM REPRESENTATION:

One approach is to ignore bytes and record a bit stream.  In fact,
the tape can be viewed as an audio recording of a modem.  The Sol
writes bytes in 10b frames: 1 start bit, 8 data bits, a two stop bits.
For 300 baud recording we'd just duplicate each bit four times,
although this isn't exactly how the real hardware behaves.  The
advantage of this scheme is regular and it allows fine-granularity
of modelling, especially inter-characacter gaps.  However, our file
format doesn't contain that low-level information, so there is no
need to model it.  Plus, it could require more complicated decoding
when reading the tape, so we will not implement that.

Instead, each 1200 baud byte on the tape is represented by a 16b short
in memory.  Recording at 300 baud causes one byte to span four of these
shorts.  The short is packed like this:

    bits [ 7: 0] -- raw data bits
    bit  [    8] -- framing error
    bit  [    9] -- overrun error
    bits [12:10] -- 000, unused
    bits [15:13] -- type:
			000=silence
			100=300 baud byte
			101=300 baud carrier
			110=1200 baud byte
			111=1200 baud carrier

Although overrun error doesn't exist on the tape per se, it allows
us to model error conditions when mimicing an original tape.

If the type is silence or carrier, bits [9:0] must all be 0.  Although
silence has no baud rate, it is probably easiest to view it as four
1200 baud byte times.

Note that each block of four shorts (relative to the beginning of
the tape) must all be the same speed (it makes splicing speeds
less of a problem).  Specifically, in 300 baud mode, each group
of four shorts consists of one data word then three 300 baud carrier
words; in 1200 baud mode, when a new block of four is written to,
all four will be set to be 1200 baud mode (with unused bytes written
to be 1200 baud carrier).

When writing to the virtual tape, the gaps between characters can
be represented in quanta of one character of the current baud rate.
That is, at 300 baud, we can't skip one short, we must skip four
shorts.  Again, this is to simplify boundary cases when changing
speeds.

CONVERTING FILE TO INTERNAL REPRESENTATION:

Converting from the file format to the internal format is not a big
problem.  The only difficulty is that the size of the file is not
known.  Reading it once to figure the size and then a second time
to actually build it isn't great in that we have to deal with dynamic
tape size anyway when we are recording a tape.  There are four
possibilities I can think of:

     1) malloc a max size tape.  A 30 minute tape would require

        (30min)*(60sec/min)*(1200baud/sec)*(char/10baud)*(2B/char)

        This is 420KB.  Most simple approach; acceptible size.

     2) Have a segmented approach.  When one segment fills, link
        it to a new segment.  Each segment can represent one minute
        or something.  Corner cases make it less attractive.

     3) realloc on overflow.  This could lead to jerky response
        on recording as each overflow could result in >400KB of
        realloc.  not acceptible.

     4) virtual memory.  Have an array of blocks.  Every access goes
        through the paging mechanism.  It is slower per access, but
        it is regular and has no uneven accesses.  acceptible.

In the worst case we need to hold all of #1 anyway, so we might
as well go for that.  I guess the down side is it limits the
maximum size tape, but it shouldn't be a problem.

OK, so constructing the internal file is determined.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "solace.h"
#include "solace_intf.h"
#include "hexfile.h"
#include "vtape.h"


// ------------- virtual tape internal representation -------------

enum { silence=0, byte300=4, carrier300=5, byte1200=6, carrier1200=7 };

// I couldn't get this to be a 2 byte packed struct.
// the compiler always made it 4 bytes, so I had to abandon it.
// typedef struct {
//     unsigned int raw:8;
//     unsigned int framing:1;
//     unsigned int overrun:1;
//     unsigned int pad:3;
//     unsigned int type:3;
// } sample_t;

#define SAMP_SHIFT_DATA     0
#define SAMP_MASK_DATA      0x00FF

#define SAMP_SHIFT_FRAMING  8
#define SAMP_MASK_FRAMING   0x0001

#define SAMP_SHIFT_OVERRUN  9
#define SAMP_MASK_OVERRUN   0x0001

#define SAMP_SHIFT_TYPE     13
#define SAMP_MASK_TYPE      0x0007

#define GET_SAMP_DATA(s)    (((s) >> SAMP_SHIFT_DATA)    & SAMP_MASK_DATA)
#define GET_SAMP_FRAMING(s) (((s) >> SAMP_SHIFT_FRAMING) & SAMP_MASK_FRAMING)
#define GET_SAMP_OVERRUN(s) (((s) >> SAMP_SHIFT_OVERRUN) & SAMP_MASK_OVERRUN)
#define GET_SAMP_TYPE(s)    (((s) >> SAMP_SHIFT_TYPE)    & SAMP_MASK_TYPE)

#define SET_SAMP_DATA(s,data) \
	( ((s)    & ~(SAMP_MASK_DATA  << SAMP_SHIFT_DATA)) + \
	 (((data) &   SAMP_MASK_DATA) << SAMP_SHIFT_DATA)  )

#define SET_SAMP_FRAMING(s,framing) \
	( ((s)       & ~(SAMP_MASK_FRAMING  << SAMP_SHIFT_FRAMING)) + \
	 (((framing) &   SAMP_MASK_FRAMING) << SAMP_SHIFT_FRAMING)  )

#define SET_SAMP_OVERRUN(s,overrun) \
	( ((s)       & ~(SAMP_MASK_OVERRUN  << SAMP_SHIFT_OVERRUN)) + \
	 (((overrun) &   SAMP_MASK_OVERRUN) << SAMP_SHIFT_OVERRUN)  )

#define SET_SAMP_TYPE(s,type) \
	( ((s)    & ~(SAMP_MASK_TYPE  << SAMP_SHIFT_TYPE)) + \
	 (((type) &   SAMP_MASK_TYPE) << SAMP_SHIFT_TYPE)  )

#define PACK_SAMP(type,overrun,framing,data) \
	( (((data)    & SAMP_MASK_DATA)    << SAMP_SHIFT_DATA) + \
	  (((framing) & SAMP_MASK_FRAMING) << SAMP_SHIFT_FRAMING) + \
	  (((overrun) & SAMP_MASK_OVERRUN) << SAMP_SHIFT_OVERRUN) + \
	  (((type)    & SAMP_MASK_TYPE)    << SAMP_SHIFT_TYPE) )


// ------------ forward references to helper functions ------------

static sample_t make_sample(int type, int overrun, int framing, int data);
static void skipwhite(char **p);
static void chop_crlf(char *p);
static int scan_hex_file(hfile_t *hf, vtape_t *vt, char *name, int baud, int exp_len, int prev_head);
static void emit_datum(vtape_t *vt, FILE *fp, int *DLen, int framing, int overrun, byte data);
static int vt_parse(vtape_t *vt, FILE *fp, char *path);
static void compute_checksum(int data, int *cksum);


// -------------------- file global variables --------------------

// magic file header
const char *g_magicstr = "SVT1: Solace Virtual Tape Format 1";

// essentially constants
static sample_t g_nada, g_car300, g_car1200;

// this applies to all tapes.  when true, baud rate is ignored on reading.
static int g_ignorebaud = 0;


// ---------------- externally visible routines ----------------

// create pointer to a blank tape
vtape_t *
vtape_create(int minutes)
{
    vtape_t *vt;

    // initialize globals-- this is as good a place as any
    g_nada    = make_sample(silence,     0, 0, 0x00);
    g_car300  = make_sample(carrier300,  0, 0, 0x00);
    g_car1200 = make_sample(carrier1200, 0, 0, 0x00);

    vt = (vtape_t *)malloc(sizeof(vtape_t));
    if (vt != NULL) {
	vt->minutes     = minutes;
	vt->maxsamples  = minutes*60*120;
	vt->sample      = (sample_t *)malloc((vt->maxsamples + 4)*sizeof(sample_t));
	vt->readonly    = 0;
	if (vt->sample == NULL) {
	    free(vt);
	    return NULL;
	}
	vtape_init(vt);
    }

    return vt;
}


// destroy tape given pointer to it.
// caller is responsible for destroying pointer.
// returns 1 on error, 0 if OK.
int
vtape_destroy(vtape_t *vt)
{
    if (vt != NULL) {
	free(vt);
	return 0;
    }
    return 1;
}


// initialize a tape
int
vtape_init(vtape_t *vt)
{
    sample_t samp = make_sample(silence, 0, 0, 0x00);
    int i;

    if (vt == NULL)
	return 1;

    vt->dirty     = 0;
    vt->label[0]  = '\0';
    vt->cursample = vt->lastsample = 0;

    for(i=0; i<(vt->maxsamples); i++)
	vt->sample[i] = samp;

    return 0;
}


int
vtape_setprop(vtape_t *vt, int prop, int value)
{
    // check globals first -- they can be set even if tape isn't valid
    if (prop == VTPROP_IGNOREBAUD) {
	// kind of dirty, but prevents having yet another access function
	g_ignorebaud = !!value;
	return SVT_OK;
    }

    if (vt == NULL)
	return SVT_BADTAPE;

    switch (prop) {
	case VTPROP_POSITION:
	    if (value >= (vt->minutes)*60*10) {
		vt->cursample = vt->lastsample = (vt->maxsamples)-1;
		return SVT_BADVALUE;
	    }
	    vt->cursample = value*12;
	    if (vt->lastsample < vt->cursample)
		vt->lastsample = vt->cursample;
	    break;
	case VTPROP_WRITEPROTECT:
	    vt->readonly = !!value;
	    vt->dirty = 1;
	    break;
	case VTPROP_DIRTY:
	    vt->dirty = !!value;
	    break;
	case VTPROP_LABEL:
	    strncpy(vt->label, (const char*)value, VT_LABELSIZE-1);
	    vt->label[VT_LABELSIZE-1] = '\0';	// just in case
	    vt->dirty = 1;
	    break;
	case VTPROP_BAUDRATE:
	default:
	    return SVT_BADPROP;
    }

    return SVT_OK;
}


int
vtape_getprop(vtape_t *vt, int prop, int *value)
{
    // check globals first -- they can be set even if tape isn't valid
    if (prop == VTPROP_IGNOREBAUD) {
	*value = g_ignorebaud;
	return SVT_OK;
    }

    if (vt == NULL) {
	if (prop == VTPROP_POSITION) {
	    // special case: empty tapes report 0 position
	    *value = 0;
	    return SVT_OK;
	}
	return SVT_BADTAPE;
    }

    switch (prop) {
	case VTPROP_POSITION:
	    *value = (vt->cursample / 12);
	    break;
	case VTPROP_WRITEPROTECT:
	    *value = vt->readonly;
	    break;
	case VTPROP_DIRTY:
	    *value = vt->dirty;
	    break;
	case VTPROP_BAUDRATE:
	    switch (GET_SAMP_TYPE(vt->sample[vt->cursample])) {
		case byte300:  case carrier300:  *value =  300; break;
		case byte1200: case carrier1200: *value = 1200; break;
		default:
		    *value = 0; break;
	    }
	    break;
	case VTPROP_LABEL:
	    *value = (int)(vt->label);
	    break;
	default:
	    return SVT_BADPROP;
    }

    return SVT_OK;
}


#if 0
// returns the next byte on tape without advancing the pointer.
// yet, or there might be errors.  we return one of the SVT_* status
// values.  curspeed is either 300 or 1200.
int
vtape_testbyte(vtape_t *vt, int curspeed, int *data)
{
    sample_t s;

    if (vt == NULL)
	return SVT_BADTAPE;

    // if the user wants to ignore the true baud rate,
    // we fudge curspeed to match what the next sample is.
    if (g_ignorebaud) {
	s = vt->sample[vt->cursample];
	switch (s.type) {
	    case carrier300:  case byte300:  curspeed =  300; break;
	    case carrier1200: case byte1200: curspeed = 1200; break;
	}
    }

    if (curspeed == 300) {
	// 300 baud

	// make sure pointer is aligned
	vt->cursample = (vt->cursample + 3) & ~3;
	if (vt->cursample >= vt->maxsamples)
	    return SVT_EOF;
	s = vt->sample[vt->cursample];
	if (s.type == byte300) {
	    *data = s.raw;
	    return (s.overrun << 1) | (s.framing << 0);
	}

    } else {
	// 1200 baud

	if (vt->cursample >= vt->maxsamples)
	    return SVT_EOF;
	s = vt->sample[vt->cursample];
	if (vt->lastsample < vt->cursample)
	    vt->lastsample = vt->cursample;
	if (s.type == byte1200) {
	    *data = s.raw;
	    return (s.overrun << 1) | (s.framing << 0);
	}
    }

    return SVT_NOBYTE;	// carrier or silence
}
#endif

// get next "sample" off tape.  however, there might not be a character
// yet, or there might be errors.  we return one of the SVT_* status
// values.  curspeed is either 300 or 1200.
int
vtape_readbyte(vtape_t *vt, int curspeed, int *data, int *overrun, int *framing)
{
    sample_t s;

    if (vt == NULL)
	return SVT_BADTAPE;

    // if the user wants to ignore the true baud rate,
    // we fudge curspeed to match what the next sample is.
    if (g_ignorebaud) {
	s = vt->sample[vt->cursample];
	switch (GET_SAMP_TYPE(s)) {
	    case carrier300:  case byte300:  curspeed =  300; break;
	    case carrier1200: case byte1200: curspeed = 1200; break;
	}
    }

    if (curspeed == 300) {
	// 300 baud

	// make sure pointer is aligned
	vt->cursample = (vt->cursample + 3) & ~3;
	if (vt->cursample >= vt->maxsamples)
	    return SVT_EOF;
	s = vt->sample[vt->cursample];
	vt->cursample += 4;
	if (vt->lastsample < vt->cursample)
	    vt->lastsample = vt->cursample;
	if (GET_SAMP_TYPE(s) == byte300) {
	    *data    = GET_SAMP_DATA(s);
	    *overrun = GET_SAMP_OVERRUN(s);
	    *framing = GET_SAMP_FRAMING(s);
	    return SVT_BYTE;
	}

    } else {
	// 1200 baud

	if (vt->cursample >= vt->maxsamples)
	    return SVT_EOF;
	s = vt->sample[vt->cursample];
	vt->cursample++;
	if (vt->lastsample < vt->cursample)
	    vt->lastsample = vt->cursample;
	if (GET_SAMP_TYPE(s) == byte1200) {
	    *data    = GET_SAMP_DATA(s);
	    *overrun = GET_SAMP_OVERRUN(s);
	    *framing = GET_SAMP_FRAMING(s);
	    return SVT_BYTE;
	}
    }

    return SVT_NOBYTE;	// carrier or silence
}


// write a byte to the current position on the tape.
// we return one of the SVT_* status codes.
//
// the flags parameter modifies what is written,
// and is one of the following SVT_* values:
//    SVT_NOBYTE, SVT_BYTE, SVT_FRAMEERR, SVT_OVERRUN, SVT_BOTHERR
//
// curspeed is either 0 (silence), 300 or 1200.
//
// the return value is one of
//    SVT_BYTE    if OK
//    SVT_EOF     if end of tape
//    SVT_BADTAPE if bad tape foramt
int
vtape_writebyte(vtape_t *vt, int curspeed, int data, int flags)
{
    sample_t s;

    if (vt == NULL)
	return SVT_BADTAPE;

    if (curspeed == 0) {
	// silence (assumed 1200 baud)

	if (vt->cursample >= vt->maxsamples)
	    return SVT_EOF;

	vt->sample[vt->cursample] = g_nada;
	vt->cursample++;

    } else if (curspeed == 300) {
	// 300 baud

	// make sure pointer is aligned
	int off = (vt->cursample & 3);
	if (off > 0) {
	    int i;
	    s = (GET_SAMP_TYPE(vt->sample[vt->cursample&~3]) == silence) ? g_nada : g_car1200;
	    for(i=off; i<4; i++) {
		vt->sample[vt->cursample] = s;
		vt->cursample++;
	    }
	}
	if (vt->cursample >= vt->maxsamples)
	    return SVT_EOF;

	if (flags < 0)
	    s = g_car300;
	else
	    s = make_sample(byte300, (flags>>1)&1, (flags&1), data);
ASSERT((vt->cursample & 3) == 0);
	vt->sample[vt->cursample + 0] = s;
	vt->sample[vt->cursample + 1] = g_car300;	// just a convention
	vt->sample[vt->cursample + 2] = g_car300;	// just a convention
	vt->sample[vt->cursample + 3] = g_car300;	// just a convention
	vt->cursample += 4;

    } else {
	// 1200 baud

	if (vt->cursample >= vt->maxsamples)
	    return SVT_EOF;

	if (flags < 0)
	    s = make_sample(carrier1200, 0, 0, 0x00);
	else
	    s = make_sample(byte1200, (flags>>1)&1, (flags&1), data);

	vt->sample[vt->cursample] = s;
	vt->cursample++;
    }

    if (vt->lastsample < vt->cursample)
	vt->lastsample = vt->cursample;

    vt->dirty = 1;

    return SVT_BYTE; // OK
}


int
vtape_fileread(vtape_t *vt, char *fname)
{
    FILE *fp;
    int r;
    char *path;

    if (vt == NULL)
	return SVT_BADTAPE;

    fp = fopen(fname, "r");
    if (fp == NULL)
	return SVT_BADFILE;	// file not readable

    // extract path to .svt file in case of indirect
    // file references with a relative path.
    path = HostExtractPath(fname);

    vtape_init(vt);
    r = vt_parse(vt, fp, path);

    fclose(fp);
    free(path);

    vt->dirty     = 0;
    vt->cursample = 0;	// rewind tape

    return r;
}


// write file from supplied virtual tape.
// this involves some amount of lookahead in order to preserve the
// high-level structure.  It would be much easier to just make one
// flat dump of D statements, but it is less useful then.

// end of file condition
#define VT_EOT(p) ((p) >= vt->lastsample)

// if we are in the middle of generating a D command, stop it
#define TERMINATE_D do { if (DLen > 0) fprintf(fp, "\n"); DLen = 0; } while(0)

int
vtape_filewrite(vtape_t *vt, char *fname)
{
    long samp_ptr;
    int samp_inc = 1;	// default: 1200 baud
    int cps      = 120;	// default: 1200 baud
    int curbaud  = 0;	// current baud rate; unknown at time 0
    int DLen = 0;	// # of chars output in D command so far
    int tenths;
    char *bp;
    FILE *fp;

    if (vt == NULL)
	return SVT_BADTAPE;

    fp = fopen(fname, "w");
    if (fp == NULL)
	return SVT_BADFILE;	// file not writable

    samp_ptr = 0;

    // print magic header
    fprintf(fp, "%s\n", g_magicstr);

    // print any labels.  lines end in \r\n
    for(bp = vt->label; *bp; ) {
	fprintf(fp, "L ");
	while (bp[0] != '\r' || bp[1] != '\n') {
	    fprintf(fp, "%c", *bp);
	    bp++;
	}
	bp += 2;
	fprintf(fp, "\n");
    }

    // if write protected, say so
    if (vt->readonly)
	fprintf(fp, "READONLY\n");

    for(;;) {

	sample_t s;
	int type;

	// check for end of tape
	if (VT_EOT(samp_ptr)) {
	    TERMINATE_D;
	    break;
	}

	// get next sample
	s = vt->sample[samp_ptr];
	type = GET_SAMP_TYPE(s);
	samp_ptr += samp_inc;

	// generate "S"ilence command as needed.
	// round up to the nearest 10th of a second.
	// silence is treated as if 1200 baud for timing.
	if (type == silence) {
	    int r = 1;
	    while (!VT_EOT(samp_ptr) && (GET_SAMP_TYPE(vt->sample[samp_ptr]) == silence)) {
		samp_ptr += samp_inc;
		r++;
	    }
	    tenths = (10*r+cps-1) / cps;
	    fprintf(fp, "\nS %d\n", tenths);
	    continue;
	}

	// check for change in baud rate; generate "B"aud command
	if ((curbaud == 0) ||
	    (curbaud == 1200 && type != byte1200 && type != carrier1200) ||
	    (curbaud ==  300 && type != byte300  && type != carrier300)) {
	    TERMINATE_D;
	    curbaud = (type == byte1200 || type == carrier1200) ? 1200 : 300;
	    fprintf(fp, "B %d\n", curbaud);
	    // change stride and alignment as necessary
	    if (curbaud == 300) {
		samp_ptr = (samp_ptr + 3) & ~3;	// round up to next 4
		samp_inc = 4;
		cps      = 30;
	    } else {
		samp_inc = 1;
		cps      = 120;
	    }
	    continue;
	}

	// generate "C"arrier command as needed.  round up to the
	// nearest 10th of a second if more than 0.5 seconds of
	// carrier is needed.  carrier is recorded at 1200 baud
	if (type == carrier300 || type == carrier1200) {
	    int r = 1;
	    while (!VT_EOT(samp_ptr) && (GET_SAMP_TYPE(vt->sample[samp_ptr]) == type)) {
		samp_ptr += samp_inc;
		r++;
	    }
	    if (r >= cps/2) {
		TERMINATE_D;
		tenths = (10*r+cps-1) / cps;
		fprintf(fp, "\nC %d\n", tenths);
		continue;
	    }
	    // we have carrier, but we just insert it into a D command
	    while (r--) {
		if (!DLen) fprintf(fp, "D ");
		fprintf(fp,"MM");
		DLen++;
		if (DLen >= 16)
		    TERMINATE_D;
	    }
	    continue;
	}

	// we must have data
	// look for an 'H' type header
	if (GET_SAMP_DATA(s) == 0x00) {

	    const int curtype = (curbaud==300) ? byte300 : byte1200;
	    int tmp_ptr = samp_ptr;
	    int nothdr = 0;
	    int r;
	    byte header[17];

	    // look for a preamble --
	    //   on write, Solos generates fifty 0x00 and one 0x01
	    //   on read, Solos looks for ten 0x00 and one 0x01
	    r = 1;
	    while(!VT_EOT(tmp_ptr)
			&& (GET_SAMP_TYPE(vt->sample[tmp_ptr]) == curtype)
			&& (GET_SAMP_DATA(vt->sample[tmp_ptr]) == 0x00)
			&& !GET_SAMP_OVERRUN(vt->sample[tmp_ptr])
			&& !GET_SAMP_FRAMING(vt->sample[tmp_ptr])) {
		r++;
		tmp_ptr += samp_inc;
	    }
	    // solos writes 50 0x00 bytes on write, but expects
	    // only 10 on read.  we'll be a little more conservative
	    // in recognizing headers on read.
	    nothdr = (r < 15);

	    // look for 0x01 after a string of 0x00
	    nothdr |= VT_EOT(tmp_ptr)
		   || (GET_SAMP_TYPE(vt->sample[tmp_ptr]) != curtype)
		   || (GET_SAMP_DATA(vt->sample[tmp_ptr]) != 0x01)
		   ||  GET_SAMP_OVERRUN(vt->sample[tmp_ptr])
		   ||  GET_SAMP_FRAMING(vt->sample[tmp_ptr]);
	    tmp_ptr += samp_inc;

	    if (!nothdr) {
		// grab next 16 bytes:
		//    5 byte file name
		//    1 byte 0x00
		//    1 byte file type
		//    2 byte length
		//    2 byte start address
		//    2 byte execute address
		//    3 byte 0x00 pad

		int cksum = 0x00;
		int i;
		for(i=0; i<17; i++) {
		    nothdr |= VT_EOT(tmp_ptr)
			   || (GET_SAMP_TYPE(vt->sample[tmp_ptr]) != curtype)
			   ||  GET_SAMP_OVERRUN(vt->sample[tmp_ptr])
			   ||  GET_SAMP_FRAMING(vt->sample[tmp_ptr]);
		    header[i] = GET_SAMP_DATA(vt->sample[tmp_ptr]);
		    tmp_ptr += samp_inc;
		    if (i < 16)
			compute_checksum(header[i], &cksum);
		    // we could enforce that bytes [13,14,15] of
		    // header[] must be 0x00, but solos doesn't either.
		}
		nothdr |= (cksum != header[16]);
	    }

	    if (!nothdr) {
		// we apparently have a header
		int i;
		int last_00 = 6;	// index to last 00 byte in name
		TERMINATE_D;
		fprintf(fp, "H ");
		// be careful -- the filename might not be simple ASCII
		for(i=5; i>1; i--) {
		    if (header[i] != '\0')
			break;
		    last_00 = i;
		}
		for(i=0; i<last_00; i++) {
		    if (header[i] <= ' ' ||
			header[i] == '\\' ||
			header[i] >= 0x7F)
			fprintf(fp, "\\%02X", (int)header[i]);
		    else
			fprintf(fp, "%c", header[i]);
		}
		fprintf(fp, " %02X %04X %04X %04X\n",
			header[6],
			header[ 7] + 256*header[ 8],
			header[ 9] + 256*header[10],
			header[11] + 256*header[12]);
		samp_ptr = tmp_ptr;
	    } else {
		// not enough of a preamble -- just emit the run of 0x00s
		int i;
		for(i=0; i<r; i++)
		    emit_datum(vt, fp, &DLen, 0,0,0x00);
		samp_ptr += samp_inc*(r-1);	// run minus one at top of loop
	    }
	    continue;

	} // (s.data == 0x00)

	// just a normal data byte
	emit_datum(vt, fp, &DLen,
		    GET_SAMP_FRAMING(s),
		    GET_SAMP_OVERRUN(s),
		    (byte)GET_SAMP_DATA(s));

    } // for(all samples)

    fclose(fp);

    vt->dirty = 0;

    return SVT_OK;
}


// ------------------------ helper functions ------------------------

static sample_t
make_sample(int type, int overrun, int framing, int data)
{
    return PACK_SAMP(type, overrun, framing, data);
}


// skip whitespace
static void
skipwhite(char **p)
{
    while (**p == ' ' || **p == '\t')
	(*p)++;
}


// remove any combination of trailing CRs and LFs from buffer
static void
chop_crlf(char *p)
{
    int len = strlen(p);
    while (p[len-1] == '\n' || p[len-1] == '\r') {
	p[len-1] = '\0';
	len--;
    }
}


// copied from SOLOS "DOCRC" routine.
// A has the data byte on entry, C has the CRC.
// *cksum should be 0 on first call.
static void
compute_checksum(int data, int *cksum)
{
    int a = data;
    int c = *cksum;

#if 1
    a -= c;			// SUB C
    c  = a;			// MOV C,A
    a ^= c;			// XRA C
    a ^= 0xFF;			// CMA
    a -= c;			// SUB C
    *cksum = (a & 0xFF);	// MOV C,A
#else
    *cksum = 0xFF & (0xFF - a + c);
#endif
}


// put out a data byte, taking care of limiting line
// length and putting out line headers.
static void
emit_datum(vtape_t *vt, FILE *fp, int *DLen, int framing, int overrun, byte data)
{
    if (!*DLen)
	fprintf(fp, "D ");
    if (framing)
	fprintf(fp, "#", data);
    if (overrun)
	fprintf(fp, "+", data);
    fprintf(fp, "%02X", data);
    (*DLen)++;
    if (*DLen >= 16) {
	fprintf(fp, "\n");
	*DLen = 0;
    }
}


// read file into supplied virtual tape.
// 0 on success, !=0 on error. if > 0, it indicates
// which line was not kosher.

// we bury this function to prevent having to fclose
// any time an error is detected.

// file header format


// return binary equivalent, or -1 on error
static int
hexval(char c)
{
    if ('0' <= c && c <= '9')
	return c - '0';
    if ('A' <= c && c <= 'F')
	return c - 'A' + 10;
    if ('a' <= c && c <= 'f')
	return c - 'a' + 10;
    return -1;
}


static int
vt_parse(vtape_t *vt, FILE *fp, char *path)
{
    int baud = 1200;	// presumed rate
    int prev_head = 0;	// indicates if previous data was from header
    char buff[256], name[256], hexpair[3], *stat;
    int line, arg, nextflags;
    struct {
	char name[6];
	char type;
	int  len;
	int  start_addr;
	int  exec_addr;
    } head;

    ASSERT(vt   != NULL);
    ASSERT(fp   != NULL);
    ASSERT(path != NULL);

    // read first line -- special case
    line = 1;
    stat = fgets(buff, sizeof(buff), fp);
    if (stat == NULL)
	return SVT_BADFILE;
    chop_crlf(buff);
    if (strcmp(buff, g_magicstr))
	return SVT_BADFILE;

    // scan file
    for(;;) {

	char *bp;
	int t, r, i;

	// get next line
	line++;
	stat = fgets(buff, sizeof(buff), fp);
	if (stat == NULL)
	    break;	// done

	// parse each line
	chop_crlf(buff);
	bp = buff;
	skipwhite(&bp);

	if (*bp == '\0')
	    continue;	// blank line

	switch (*bp) {

	    case ';':	// comment
		break;


	    case 'L':	// label
		bp++;
		skipwhite(&bp);
		r = strlen(vt->label);
		t = strlen(bp);
		if (r+t+1 < sizeof(vt->label)) {
		    // append label
		    strcpy( (vt->label)+r,   bp );
		    strcpy( (vt->label)+r+t, "\r\n" );
		}
		break;


	    case 'R':	// read-only
		if (!strcmp(bp, "READONLY"))
		    vt->readonly = 1;
		else
		    return line;
		break;


	    case 'B':	// baud rate
		bp++;
		skipwhite(&bp);
		if (!strcmp(bp, "300"))
		    baud = 300;
		else if (!strcmp(bp, "1200"))
		    baud = 1200;
		else
		    return line;
		prev_head = 0;
		break;


	    case 'S':	// silence
		bp++;
		if (sscanf(bp, " %d", &arg) != 1)
		    return line;
		// generate arg 10ths of a second of silence on the tape
		// writebyte treats silence like 1200 baud
		for(i=0; i < arg*12; i++) {
		    r = vtape_writebyte(vt, 0, 0x00, SVT_NOBYTE);
		    if (r == SVT_BYTE)
			continue;
		    if (r != SVT_EOF)
			break;	// tape full
		    return r;	// propagate error
		}
		break;


	    case 'C':	// carrier
		bp++;
		if (sscanf(bp, " %d", &arg) != 1)
		    return line;
		// generate arg 10ths of a second of carrier on the tape
		for(i=0; i < arg * ((baud == 300) ? 3:12); i++) {
		    r = vtape_writebyte(vt, baud, 0x00, SVT_NOBYTE);
		    if (r == SVT_BYTE)
			continue;
		    if (r != SVT_EOF)
			break;	// tape full
		    return r;	// propagate error
		}
		break;


	    case 'D':	// data

		nextflags = 0;
		hexpair[2] = '\0';
		bp++;

		for(;;) {

		    skipwhite(&bp);
		    if (*bp == '\0')
			break;

		    // accumulate error flags
		    if (*bp == '#') {
			nextflags |= SVT_FRAMEERR;
			bp++;
			continue;
		    } else if (*bp == '+') {
			nextflags |= SVT_OVERRUN;
			bp++;
			continue;
		    }

		    // get two characters
		    hexpair[0] = *bp++;
		    hexpair[1] = *bp++;
		    if (!hexpair[1])
			return line;	// not an even pair

		    // look for "magic hex" carrier symbol
		    if (!strcmp(hexpair, "MM")) {
			if (nextflags != 0)
			    return line;
			r = vtape_writebyte(vt, baud, 0x00, SVT_NOBYTE);
			if (r != SVT_BYTE)
			    return r;	// propagate error
			continue;
		    }

		    if (sscanf(hexpair, "%2x", &arg) != 1)
			return line;

		    r = vtape_writebyte(vt, baud, arg, nextflags);
		    if (r != SVT_BYTE)
			return r;	// propagate error
		    nextflags = 0;	// in preparation for next time
		    continue;

		} // for(;;) next character of line
		prev_head = 0;
		break;


	    case 'H':	// header
		{
		    int cksum, np;
		    char litname[256];
		    byte header[17];

		    if (sscanf(bp, "H %s %2x %4x %4x %4x",
				litname, &head.type, &head.len,
				&head.start_addr, &head.exec_addr) != 5)
			return line;

		    // parse file name; it might not be simple ASCII
		    np = 0;
		    for(i=0; i<5; i++) {
			head.name[i] = litname[np];	// including null
			if (litname[np] == '\0')
			    break;
			if (litname[np] != '\\') {
			    np++;
			} else {
			    // escaped character
			    int val;
			    if ((strlen(&litname[np])  < 3) ||
				(hexval(litname[np+1]) < 0) ||
				(hexval(litname[np+2]) < 0) )
				return line;
			    val = 16*hexval(litname[np+1])
				+    hexval(litname[np+2]);
			    np += 3;
			    head.name[i] = val;
			}
		    }
		    if (litname[np] != '\0')
			return line;		// extra chars at end of name?
		    head.name[5] = '\0';	// just in case

		    // check parameters
		    if ( (strlen(head.name) > 5)    ||
			 (head.type       > 0xFF)   ||
			 (head.len        > 0xFFFF) ||
			 (head.start_addr > 0xFFFF) ||
			 (head.exec_addr  > 0xFFFF))
			return line;

		    // pad name with nulls
		    for(i=strlen(head.name); i<5; i++)
			head.name[i] = '\0';

		    // write preamble: fifty 0x00s, one 0x01
		    r = 0;
		    for(i=0; i<50; i++)
			r |= vtape_writebyte(vt, baud, 0x00, SVT_BYTE);
		    r |= vtape_writebyte(vt, baud, 0x01, SVT_BYTE);
		    // write standard 16 byte header
		    for(i=0; i<5; i++)
			header[i] = head.name[i];
		    header[ 5] =  0x00;
		    header[ 6] =  head.type;
		    header[ 7] = (head.len & 0xFF);
		    header[ 8] = (head.len >> 8);
		    header[ 9] = (head.start_addr & 0xFF);
		    header[10] = (head.start_addr >> 8);
		    header[11] = (head.exec_addr  & 0xFF);
		    header[12] = (head.exec_addr  >> 8);
		    header[13] =
		    header[14] =
		    header[15] = 0x00;
		    cksum = 0x00;
		    for(i=0; i<16; i++)
			compute_checksum(header[i], &cksum);
		    header[16] = cksum;
		    for(i=0; i<17; i++) {
			r = vtape_writebyte(vt, baud, header[i], SVT_BYTE);	// 5-byte name
			if (r != SVT_OK)
			    break;
		    }
		    if (r != 0)
			return r;
		}
		prev_head = 1;	// expect a file to follow
		break;


	    case 'F':	// file
		{
		    int format, status;
		    char *fullname;
		    hfile_t *hf;
		    if (sscanf(buff, "F %s", name) != 1)
			return line;
		    fullname = (char*)malloc(strlen(name) + strlen(path) + 2);
		    ASSERT(fullname);
		    if (HostIsAbsolutePath(name)) {
			strcpy(fullname, name);
		    } else {
			strcpy(fullname, path);
			strcat(fullname, "\\");
			strcat(fullname, name);
		    }
		    buff[0] = '\0';
		    format = hexfile_guessformat(fullname);
		    if (format < 0)
			sprintf(buff, "Error: virtual tape refers to unreadable file '%s'", name);
		    else if (format > 1)
			sprintf(buff, "Error: virtual tape refers to '%s' with unknown filetype", name);
		    else {
			hf = hexfile_hex_open(fullname, format, "r");
			if (hf == NULL)
			    sprintf(buff, "Error: virtual tape refers to unreadable file '%s'", name);
		    }
		    if (strlen(buff) > 0) {
			UI_Alert(buff);
			return SVT_BADFILE;
		    }
		    status = scan_hex_file(hf, vt, fullname, baud, head.len, prev_head);
		    hexfile_hex_close(hf);
		    free(fullname);
		    if (status != SVT_OK)
			return status;
		}
		break;


	    default:
		// not understood
		return line;
	} // switch

    } // for(;;)

    return SVT_OK;	// OK format
}


// scan_hex_file is called as a separate function so that we don't have
// to keep closing & cleaning up in each place where we detect an error.
// if prev_head is set, we check the length consistency between what the
// header claims and what the file actually contains.
// return SVT_OK on success, SVT_BADFILE on failure.

// globals used by scan_hex_file and helper function
static int shf_len, shf_cksum, shf_blocklen;

// helper to scan_hex_file
// emits one byte and keeps track of checksum and block boundaries.
// call with data = -1 at end to emit any final checksum.
static int
append_byte(vtape_t *vt, char *name, int baud, int data)
{
    int status;
    int final_checksum = (data < 0);

    if (!final_checksum) {
	status = vtape_writebyte(vt, baud, data, SVT_BYTE);
	if (status != SVT_BYTE) {
	    UI_Alert("Error creating virtual tape while reading '%s'", name);
	    return SVT_BADFILE;
	}

	if (shf_blocklen == 0)
	    shf_cksum = 0x00;
	compute_checksum(data, &shf_cksum);

	shf_blocklen++;
	shf_len++;
    }

    if ((shf_blocklen == 256) || (final_checksum && (shf_blocklen > 0))) {
	// emit checksum byte
	status = vtape_writebyte(vt, baud, shf_cksum, SVT_BYTE);
	if (status != SVT_BYTE) {
	    UI_Alert("Error creating virtual tape while reading '%s'", name);
	    return SVT_BADFILE;
	}
	shf_blocklen = 0;
    }

    return SVT_OK;
}


static int
scan_hex_file(hfile_t *hf, vtape_t *vt, char *name, int baud, int exp_len, int prev_head)
{
    int addr, prevaddr, any, status;
    byte data;

    shf_len = 0;	// # bytes in file
    shf_blocklen = 0;
    prevaddr = -1;

    any = 0;
    status = hexfile_hex_read(hf, &addr, &data);

    while (status == HFILE_OK) {

	if (any && (addr <= prevaddr)) {
	    UI_Alert("Error: non-monotonic addressing in '%s'", name);
	    return SVT_BADFILE;
	}

	// fill gaps with 0x00
	prevaddr++;
	while (any && (prevaddr < addr)) {
	    status = append_byte(vt, name, baud, 0x00);
	    if (status != SVT_BYTE)
		return SVT_BADFILE;
	    prevaddr++;
	}

	status = append_byte(vt, name, baud, data);
	if (status != SVT_BYTE)
	    return SVT_BADFILE;

	prevaddr = addr;
	any = 1;

	status = hexfile_hex_read(hf, &addr, &data);

    } // while

    if (status != HFILE_EOF) {
	UI_Alert("Error: Bad format in '%s'", name);
	return SVT_BADFILE;
    }

    // emit checksum -- final block might be less than 256 bytes
    status = append_byte(vt, name, baud, -1);
    if (status != SVT_BYTE)
	return SVT_BADFILE;

    if (prev_head && (exp_len != shf_len)) {
	UI_Alert("Warning: Actual length of file\n"
		 "    '%s'\n"
		 "is different than what preceeding H header expects", name);
    }

    return SVT_OK;
}

